/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpKeySpec.c,v 1.3 2000/10/01 22:54:14 hal Exp $
____________________________________________________________________________*/
#include "pgpConfig.h"
#include <time.h>
#include <string.h>
#include "pgpKeySpec.h"
#include "pgpMem.h"
#include "pgpEnv.h"
#include "pgpErrors.h"
#include "pgpTimeDate.h"
#include "pgpUsuals.h"

/*
 * The things in a public key that aren't part of the (algorithm-specific)
 * mathematical public key.  Personally, I'd prefer that this didn't exist!
 */
struct PGPKeySpec
{
	PGPContextRef	context;
	PgpVersion version;			/* Version of key to create */
	PGPUInt32 creation;			/* Creation date */
	PGPUInt16 expirationDays;	/* Validity period */
	PGPByte pkalg;				/* Public key algorithm */
	PGPUInt32 keyflags;			/* Keyflags */
	PGPByte fkeyflags;			/* Have keyflags set */
	PGPUInt32 keyservprefs;		/* Key serv preference mask */
	PGPByte fkeyservprefs;		/* Have keyservprefs set */
	PGPByte const *prefkeyserv;	/* Preferred key server */
	PGPSize prefkeyservlength;
	PGPByte const *prefalgs;	/* Preferred PK algorithms */
	PGPSize prefalgslength;

	DEBUG_STRUCT_CONSTRUCTOR( PGPKeySpec )
} ;

PGPKeySpec *
pgpKeySpecCreate(PGPEnv const *env)
{
	PGPKeySpec *ks;
	PGPContextRef		cdkContext	= pgpenvGetContext( env );

	ks = (PGPKeySpec *)pgpContextMemAlloc( cdkContext,
		sizeof(*ks), kPGPMemoryMgrFlags_Clear);
	if (ks) {
		ks->context	= cdkContext;
		/* Default values, based on environment */
		/*ks->version = pgpenvGetInt(env, PGPENV_VERSION, NULL, NULL);*/
		ks->version = PGPVERSION_3;	/* Default to this unless overridden */
		ks->creation = pgpTimeStamp(pgpenvGetInt(env, PGPENV_TZFIX,
		                                         NULL, NULL));
		ks->expirationDays = 0;	/* Forever */
	}
	return ks;
}

PGPKeySpec *
pgpKeySpecCopy(PGPKeySpec const *ks)
{
	PGPKeySpec *ks2;

	ks2 = (PGPKeySpec *)pgpContextMemAlloc( ks->context,
		sizeof(*ks2), kPGPMemoryMgrFlags_Clear);
	if( IsNull( ks2 ) )
		return NULL;
	pgpCopyMemory( ks, ks2, sizeof(*ks2) );
	if( IsntNull( ks->prefkeyserv ) )
	{
		ks2->prefkeyserv = pgpContextMemAlloc( ks->context,
											   ks->prefkeyservlength, 0 );
		if( IsNull( ks2->prefkeyserv ) )
		{
			PGPFreeData( ks2 );
			return NULL;
		}
		pgpCopyMemory( ks->prefkeyserv, (PGPByte *)ks2->prefkeyserv,
					   ks->prefkeyservlength );
	}
	if( IsntNull( ks->prefalgs ) )
	{
		ks2->prefalgs = pgpContextMemAlloc( ks->context,
											ks->prefalgslength, 0 );
		if( IsNull( ks2->prefalgs ) )
		{
			if( IsntNull( ks2->prefkeyserv ) )
				PGPFreeData( (PGPByte *)ks2->prefkeyserv );
			PGPFreeData( ks2 );
			return NULL;
		}
		pgpCopyMemory( ks->prefalgs, (PGPByte *)ks2->prefalgs,
					   ks->prefalgslength );
	}
	return ks2;
}

void
pgpKeySpecDestroy(PGPKeySpec *ks)
{
	pgpAssertAddrValid( ks, PGPKeySpec );

	if( IsntNull( ks->prefkeyserv ) )
		PGPFreeData( (PGPByte *)ks->prefkeyserv );
	if( IsntNull( ks->prefalgs ) )
		PGPFreeData( (PGPByte *)ks->prefalgs );
	PGPFreeData( ks);
}

PgpVersion
pgpKeySpecVersion(PGPKeySpec const *ks)
{
	return ks->version;
}

PGPUInt32
pgpKeySpecCreation(PGPKeySpec const *ks)
{
	return ks->creation;
}

PGPUInt16
pgpKeySpecValidity(PGPKeySpec const *ks)
{
	return ks->expirationDays;
}

PGPByte
pgpKeySpecPkAlg( PGPKeySpec const *ks )
{
	return ks->pkalg;
}

PGPUInt32
pgpKeySpecKeyflags( PGPKeySpec const *ks, PGPBoolean *fkeyflags )
{
	*fkeyflags = ks->fkeyflags;
	return ks->keyflags;
}

PGPUInt32
pgpKeySpecKeyservPrefs( PGPKeySpec const *ks, PGPBoolean *fkeyservprefs )
{
	*fkeyservprefs = ks->fkeyservprefs;
	return ks->keyservprefs;
}

PGPByte const *
pgpKeySpecPrefKeyserv( PGPKeySpec const *ks, PGPSize *length )
{
	*length = ks->prefkeyservlength;
	return ks->prefkeyserv;
}

PGPByte const *
pgpKeySpecPrefAlgs( PGPKeySpec const *ks, PGPSize *length )
{
	*length = ks->prefalgslength;
	return ks->prefalgs;
}

int
pgpKeySpecSetVersion(PGPKeySpec *ks, PgpVersion ver)
{
	switch(ver) {
	  case PGPVERSION_2:
	  case PGPVERSION_3:
	  case PGPVERSION_4:
		ks->version = ver;
		return kPGPError_NoErr;
	}
	/* default */
	return kPGPError_UnknownKeyVersion;
}

int
pgpKeySpecSetCreation(PGPKeySpec *ks, PGPUInt32 creation)
{
	ks->creation = creation;
	return kPGPError_NoErr;
}

int
pgpKeySpecSetValidity(PGPKeySpec *ks, PGPUInt16 expirationDays )
{
	ks->expirationDays = expirationDays;
	return kPGPError_NoErr;
}

int
pgpKeySpecSetPkAlg(PGPKeySpec *ks, PGPByte pkalg )
{
	ks->pkalg = pkalg;
	return kPGPError_NoErr;
}

int
pgpKeySpecSetKeyflags(PGPKeySpec *ks, PGPUInt32 keyflags)
{
	ks->fkeyflags = TRUE;
	ks->keyflags = keyflags;
	return kPGPError_NoErr;
}

int
pgpKeySpecSetKeyservPrefs(PGPKeySpec *ks, PGPUInt32 keyservprefs)
{
	ks->fkeyservprefs = TRUE;
	ks->keyservprefs = keyservprefs;
	return kPGPError_NoErr;
}

int
pgpKeySpecSetPrefKeyserv(PGPKeySpec *ks, PGPByte const *prefkeyserv,
						 PGPSize length )
{
	if( length == 0 || IsNull( prefkeyserv ) )
		return kPGPError_NoErr;
	if( IsntNull( ks->prefkeyserv ) )
	{
		PGPFreeData( (PGPByte *)ks->prefkeyserv );
		ks->prefkeyserv = NULL;
	}
	ks->prefkeyserv = pgpContextMemAlloc( ks->context, length, 0 );
	if( IsNull( ks->prefkeyserv ) )
		return kPGPError_OutOfMemory;
	pgpCopyMemory( prefkeyserv, (PGPByte *)ks->prefkeyserv, length);
	ks->prefkeyservlength = length;
	return kPGPError_NoErr;
}

int
pgpKeySpecSetPrefAlgs(PGPKeySpec *ks, PGPByte const *prefalgs,
						 PGPSize length )
{
	if( length == 0 || IsNull( prefalgs ) )
		return kPGPError_NoErr;
	if( IsntNull( ks->prefalgs ) )
	{
		PGPFreeData( (PGPByte *)ks->prefalgs );
		ks->prefalgs = NULL;
	}
	ks->prefalgs = pgpContextMemAlloc( ks->context, length, 0 );
	if( IsNull( ks->prefalgs ) )
		return kPGPError_OutOfMemory;
	pgpCopyMemory( prefalgs, (PGPByte *)ks->prefalgs, length);
	ks->prefalgslength = length;
	return kPGPError_NoErr;
}



/* Serializing keyspec */

PGPError
pgpKeySpecFlatten( PGPKeySpec const *ks, PGPByte **pbuf, PGPSize *plen )
{
	PGPContextRef context = ks->context;
	PGPUInt32 *buf;
	PGPSize len;

	len = 10 * sizeof(PGPUInt32) + ks->prefkeyservlength + ks->prefalgslength;
	buf = (PGPUInt32 *)pgpContextMemAlloc( context, len, 0 );
	if( IsNull( buf ) )
		return kPGPError_OutOfMemory;
	buf[0] = ks->version;
	buf[1] = ks->creation;
	buf[2] = ks->expirationDays;
	buf[3] = ks->pkalg;
	buf[4] = ks->keyflags;
	buf[5] = ks->fkeyflags;
	buf[6] = ks->keyservprefs;
	buf[7] = ks->fkeyservprefs;
	buf[8] = (PGPUInt32) ks->prefkeyservlength;
	buf[9] = (PGPUInt32) ks->prefalgslength;
	if( IsntNull( ks->prefkeyserv ) )
		pgpCopyMemory( ks->prefkeyserv, buf+10, ks->prefkeyservlength );
	if( IsntNull( ks->prefalgs ) )
		pgpCopyMemory( ks->prefalgs, (PGPByte *)(buf+10)+ks->prefkeyservlength,
					   ks->prefalgslength);

	*pbuf = (PGPByte *)buf;
	*plen = len;
	return kPGPError_NoErr;
}


PGPKeySpec *
pgpKeySpecFromBuf( PGPContextRef context, PGPByte const *bbuf, PGPSize len,
				   PGPError *err )
{
	PGPKeySpec *ks;
	PGPUInt32 *buf = (PGPUInt32 *)bbuf;

	*err = kPGPError_NoErr;
	(void) len;

	ks = (PGPKeySpec *)pgpContextMemAlloc( context,
		sizeof(*ks), kPGPMemoryMgrFlags_Clear);
	if( IsNull( ks ) )
	{
		*err = kPGPError_OutOfMemory;
		return NULL;
	}

	ks->context = context;
	ks->version = (PgpVersion) buf[0];
	ks->creation = buf[1];
	ks->expirationDays = (PGPUInt16) buf[2];
	ks->pkalg = (PGPByte) buf[3];
	ks->keyflags = buf[4];
	ks->fkeyflags = (PGPBoolean) buf[5];
	ks->keyservprefs = buf[6];
	ks->fkeyservprefs = (PGPBoolean) buf[7];
	ks->prefkeyservlength = (PGPSize)buf[8];
	ks->prefalgslength = (PGPSize)buf[9];
	if( ks->prefkeyservlength > 0 )
	{
		ks->prefkeyserv = pgpContextMemAlloc( context, ks->prefkeyservlength,
											  0 );
		if( IsNull( ks->prefkeyserv ) )
		{
			PGPFreeData( ks );
			*err = kPGPError_OutOfMemory;
			return NULL;
		}
		pgpCopyMemory( buf+10, (PGPByte *)ks->prefkeyserv,
					   ks->prefkeyservlength );
	}
	if( ks->prefalgslength > 0 )
	{
		ks->prefalgs = pgpContextMemAlloc( context, ks->prefalgslength,
											  0 );
		if( IsNull( ks->prefalgs ) )
		{
			if( IsntNull( ks->prefkeyserv ) )
				PGPFreeData( (PGPByte *)ks->prefkeyserv );
			PGPFreeData( ks );
			*err = kPGPError_OutOfMemory;
			return NULL;
		}
		pgpCopyMemory( (PGPByte *)(buf+10)+ks->prefkeyservlength,
					   (PGPByte *)ks->prefalgs, ks->prefalgslength );
	}
	return ks;
}
